package WebConfig;

our(@ISA, @EXPORT_OK);

use strict;
use Exporter;

@ISA = ('Exporter');
our @EXPORT_OK = qw( 
	Add_gene
	CGI_init
	DB_init
	Get_gene_key
	Map_all_ids
	Name_type
	Name_types
	Map_id_to_gene
	);
our @EXPORT = qw( Form_error );

#---------------------------
#Written by Jim Lund, Department of Biology, University of Kentucky.
#jiml@uky.edu
#
#Web site: http://elegans.uky.edu
#---------------------------
#
#Created: 7-05
#Last Modified: 7-22-05
#Version: .001
#
#
#HISTORY
#
#v.001, 7-05
#-Added CGI_init DB_init Set_new Get_set Set_add Get_all_sets Get_perms Get_perm Set_session Goto_main_page.
#
#




#
#Gene List web site routines.
#


############################################################################
#Calls: none
#
#Input: Reads from config file gl_config.txt in main Gene Lists directory.
#Output: Sets a bunch of config variables.
#
#Reads from the config file, and interprets each line a "variable = value" pair.
#The variable is set as a global value for the CGI.
#
sub CGI_init {

my $config_file="config.txt";
use vars qw( $variable $value );


open(FILE,'<'.$config_file) or warn "Can't open file $config_file: $!\n";

while(<FILE>) {
  chomp;

#
#Skip blank and comment lines.
#
  s/(^\s+|\s+$)//g;
  if (!$_ || /^#/) { next; }

#
#Grab variable/value pair.
#
  s/^\s*([^\s]+?)\s*=\s*//;
  my $variable = $1;

  my $value = $_; 


#
#Set variable as a Perl var.
#
  no strict;

  my $eval=q/$/.$variable.q/ = /.$value;
  eval $eval;
  push(@EXPORT,'$'.$variable);
  WebConfig->export_to_level(1, @EXPORT);
  WebConfig->export_to_level(0, @EXPORT);
  use strict;
}

close FILE or warn "Can't close file $config_file: $!\n";
}



############################################################################
#Calls: none
#
#Input: None.
#Output: Returns database handle, DB-specific way to set current date.
#
#Open a DB connection and set DB-specific variables.
#
sub DB_init {

use DBI;
use vars qw( $dbh $DB_DATE $DB_TYPE $dsn $user $password );

#
#Initailize db connection.
#
if ($DB_TYPE eq 'mysql') {

  $dsn="DBI:mysql:gl";
  $DB_DATE='NULL';

} elsif ($DB_TYPE eq 'oracle') {

  $dsn="DBI:Oracle:ORCL";
  $DB_DATE='sysdate';

}

$dbh = DBI->connect($dsn, $user, $password,
    {PrintError => 1, RaiseError => 1})  || die $DBI::errstr;


return($dbh,$DB_DATE);
}



############################################################################
#Input: error string.
#
#Output: HTML page containg the error string.  Exits the program.
#
#Print out error message and exits.
#
sub Form_error {
my ($error)=@_;

my $q = new CGI;
print $q->header();

print "<HTML><HEAD><TITLE>Error</TITLE></HEAD>\n";
print "<BODY BGCOLOR='#EBE0FF'><H1>Error</H1><P><P>\n";
print $error;
print "\n<P></BODY></HTML>\n";
exit;
}



############################################################################
#Input: none
#
#Output: Hash of gene name mapping type and associated name_type_ids
#
#Returns gene name mapping types and associated name_type_ids.
#
sub Name_types {

my $search = "SELECT descr,name_type FROM name_type";

my $name_types = $dbh->selectall_hashref($search, 'descr');

return $name_types;
}



############################################################################
#Input: none
#
#Output: Hash of gene name mapping type and associated name_type_ids
#
#Returns gene name mapping types and associated name_type_ids.
#
sub Name_type {
my ($name_type_id) = @_;

my $search = "SELECT descr FROM name_type WHERE name_type=?";

my @row = $dbh->selectrow_array($search, undef, $name_type_id);

return $row[0];
}



############################################################################
#Input: gene name, optionally a gene_name_key
#
#Output: the new gene_name_key
#
#Adds a new gene identifier to gene and gene_all_names.  If a gene_name_key is 
#given, it is used.  Otherwise a new key is assigned.
#
#Calls XX to log the addition.
#
sub Add_gene {
my ($novel_gene_ref,$gene_name_key) = @_;


#
#Get the new gene_name_key.
#
if (!$gene_name_key) {
  my $search = "SELECT max(gene_name_key) FROM gene";

  my @row = $dbh->selectrow_array($search);
  $gene_name_key = $row[0] + 1;
}



#
#Insert into gene.
#
my $gene_cos_name = $$novel_gene_ref[0];
my $all_genes = join(';',@$novel_gene_ref);

my $search = "INSERT INTO gene VALUES (?,?,?,?,$DB_DATE)";
my $sth = $dbh->prepare($search);
$sth->execute($gene_name_key,$gene_cos_name,$gene_cos_name,$all_genes);



#
#Add the new gene(s) to gene_all_names.
#
my $search = "INSERT INTO gene_all_names VALUES (?,?,?,?,?,$DB_DATE)";

$sth= $dbh->prepare($search);
foreach my $gene (@$novel_gene_ref) {
  $sth->execute($gene,$gene_name_key,0,0,0);
}


return($gene_name_key);
}



############################################################################
#Input: a gene id
#
#Output: the gene_name_key, or 0 if the gene_id is not in table gene.
#
#
sub Get_gene_key {
my ($gene_id) = @_;


#
#Get the new gene_name_key.
#
my $search = "SELECT gene_name_key FROM gene where gene_cos_name=?";
my @row = $dbh->selectrow_array($search,undef,$gene_id);

if (!$row[0]) { $row[0] = 0; }


return $row[0];
}



############################################################################
#Input: Hash of gene ids, gene id type
#
#Output: none, the hash passed in has a 'wb_gene' sub-hash added.
#
#Map gene ids to current WormBase cosmid names.
#
sub Map_id_to_gene {
my ($gene,$map_type_id) = @_;

my $search = "SELECT n.wb_name FROM name_to_gene n, gene g WHERE n.name_type=$map_type_id AND n.gene_name_key = g.gene_name_key AND gene_cos_name=?";
my $sth= $dbh->prepare($search);

foreach my $gene_id (keys %$gene) {
  my @row = $dbh->selectrow_array($sth,undef,$gene_id);
  $$gene{$gene_id}{'wb_gene'} = $row[0];
}
}



############################################################################
#Input: none
#
#Output: Hash of gene_cos_names (gene_ids), sub-hash name_type with 'wb_gene' values.
#
#Map gene ids to current WormBase cosmid names.
#
sub Map_all_ids {

my ($dbh,$DB_DATE) = DB_init;

my %id_to_gene;

my $search = "SELECT n.wb_name, n.name_type, g.gene_cos_name FROM name_to_gene n, gene g WHERE n.gene_name_key = g.gene_name_key";

my $row_refs = $dbh->selectall_arrayref($search);

foreach my $row_ref (@$row_refs) {
  my ($wb_name, $name_type, $gene_id) = @$row_ref;
  $id_to_gene{$gene_id}{$name_type} = $wb_name;
}

return(\%id_to_gene);
}


1;
